* Title: 	bartik.do
* Version: 	23 May 2018
* Purpose: 	Produce Bartik shocks for ASEC, ACS, and Census/ACS data

*******************************************************************************
* (0) Start of file
*******************************************************************************

capture log close
log using log/bartik, replace
set more off
clear all
macro drop _all

foreach dataset in asec acs census {

	if "`dataset'" == "asec" {
		local baseyear 1977
		local geog "statefip"
		local weight "wtsupp"
	}

	if "`dataset'" == "acs" {
		local baseyear 2005
		local geog "cpuma0010"
		local weight "perwt"
	}

	if "`dataset'" == "census" {
		local baseyear 1980
		local geog "conspuma"
		local weight "perwt"
	}	
	
	use year statefip `geog' EmpB IndustryI `weight' using dta/`dataset'_clean, clear
	keep if year>=`baseyear'
	
	* Keep only continental 48 states (excluding District of Columbia)
	drop if (statefip == 02 | statefip == 11 | statefip == 15 | statefip>56)

	* Fix coding due to post-2006 PUMA change due to Katrina
	if "`dataset'" == "census" replace conspuma = 172 if conspuma == 173

	*******************************************************************************
	* (1) Industry employment by state/PUMA
	*******************************************************************************

	assert (!mi(IndustryI) & EmpB==1) | (mi(IndustryI) & EmpB==0)
	keep if EmpB==1

	* For combined Census/ACS file, drop 2006-2008 ACS and pool data for 2009-2011 ACS
	if "`dataset'" == "census" {
		keep if year == 1980 | year == 1990 | year == 2000 | (year>=2009 & year<=2011)
		replace year = 2010 if year>=2009 & year<=2011
	}
	
	collapse (sum) Emp = EmpB [pw = `weight'], by(IndustryI `geog' year)
	if "`dataset'" == "census" replace Emp = (1/3) * Emp if year==2010

	reshape wide Emp, i(`geog' year) j(IndustryI)
	forvalues j=1/12 {
		replace Emp`j' = 0 if Emp`j' == .
	}	
	reshape long Emp, i(`geog' year) j(IndustryI)

	bysort `geog' year: egen double TotalEmp = total(Emp)
	
	* Generate base-year constant industry shares
	preserve
	keep if year == `baseyear'
	reshape wide Emp TotalEmp, i(`geog' year) j(IndustryI)
	forvalues j=1/12 {
		gen double Share`baseyear'_`j' = Emp`j' / TotalEmp`j'
	}	
	keep `geog' Share`baseyear'*
	tempfile share
	save `share'

	* Generate annual changes in employment
	restore
	bysort IndustryI year: egen TotalEmpNational = sum(Emp)
	gen double ExogenousEmp = TotalEmpNational - Emp

	reshape wide Emp TotalEmp ExogenousEmp TotalEmpNational, i(`geog' year) j(IndustryI)
	merge m:1 `geog' using `share', assert(matched) nogen
	

	*******************************************************************************
	* (2) Generate Bartik shocks and save data
	*******************************************************************************

	xtset `geog' year
	forvalues j=1/12 {
		if "`dataset'" == "census" 	gen double Bartik`j' = Share`baseyear'_`j' * ((ExogenousEmp`j' - L10.ExogenousEmp`j') / L10.ExogenousEmp`j')
		else 						gen double Bartik`j' = Share`baseyear'_`j' * ((ExogenousEmp`j' - L.ExogenousEmp`j') / L.ExogenousEmp`j')
	}	

	keep year `geog' Bartik*
	reshape long Bartik, i(year `geog') j(IndustryI)
	collapse (sum) Bartik, by(year `geog')
	
	compress
	save dta/bartik_`dataset', replace
}	


*******************************************************************************
* (3) End of file
*******************************************************************************

log close
exit, clear


